<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\UserProfile;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class UserController extends Controller
{
    public function getProfile(Request $request): JsonResponse
    {
        try {
            $user = $request->user();
            $user->load('profile');

            return response()->json([
                'success' => true,
                'data' => [
                    'user' => $user,
                    'profile' => $user->profile,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve profile',
                'error_code' => 'PROFILE_001'
            ], 500);
        }
    }

    public function updateProfile(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'name' => 'nullable|string|max:255',
                'preferred_language' => 'nullable|in:en,ms',
                'default_industry' => 'nullable|string|max:255',
            ]);

            $user = $request->user();

            // Update user name if provided
            if (isset($validated['name'])) {
                $user->update(['name' => $validated['name']]);
            }

            // Update or create profile
            $profileData = collect($validated)
                ->only(['preferred_language', 'default_industry'])
                ->filter()
                ->toArray();

            if (!empty($profileData)) {
                UserProfile::updateOrCreate(
                    ['user_id' => $user->id],
                    $profileData
                );
            }

            $user->load('profile');

            return response()->json([
                'success' => true,
                'data' => [
                    'user' => $user,
                    'profile' => $user->profile,
                ],
                'message' => 'Profile updated successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update profile',
                'error_code' => 'PROFILE_002'
            ], 500);
        }
    }
}
