import React from 'react';
import { LearningStyle as LearningStyleType } from '../../services/api';
import { BookOpen, Eye, Headphones, Monitor, Users, Zap } from 'lucide-react';

interface LearningStyleProps {
  style: LearningStyleType;
  rootNumber: number;
}

const LearningStyle: React.FC<LearningStyleProps> = ({ style, rootNumber }) => {
  const { style: styleName, approach } = style;

  const getStyleIcon = (styleName: string) => {
    if (styleName.includes('KINESTETIK')) return Users;
    if (styleName.includes('VISUAL')) return Eye;
    if (styleName.includes('AUDIO')) return Headphones;
    if (styleName.includes('DIGITAL')) return Monitor;
    if (styleName.includes('FLEXIBLE')) return Zap;
    return BookOpen;
  };

  const getStyleColor = (styleName: string) => {
    if (styleName.includes('KINESTETIK')) return 'from-green-500 to-teal-500';
    if (styleName.includes('VISUAL')) return 'from-blue-500 to-cyan-500';
    if (styleName.includes('AUDIO')) return 'from-purple-500 to-pink-500';
    if (styleName.includes('DIGITAL')) return 'from-gray-500 to-slate-500';
    if (styleName.includes('FLEXIBLE')) return 'from-yellow-500 to-orange-500';
    return 'from-indigo-500 to-purple-500';
  };

  const getStyleTips = (styleName: string) => {
    if (styleName.includes('KINESTETIK')) {
      return [
        'Learn by doing and practicing hands-on activities',
        'Take breaks to move around during study sessions',
        'Use physical objects and manipulatives when possible',
        'Engage in role-playing and real-world simulations',
        'Work in groups for collaborative learning experiences'
      ];
    }
    if (styleName.includes('VISUAL')) {
      return [
        'Use charts, diagrams, and visual aids to understand concepts',
        'Create mind maps and flowcharts for complex information',
        'Organize information using colors and highlighting',
        'Watch videos and demonstrations before attempting tasks',
        'Keep your workspace clean and visually organized'
      ];
    }
    if (styleName.includes('AUDIO')) {
      return [
        'Listen to recordings and podcasts related to your topics',
        'Participate in discussions and verbal explanations',
        'Read aloud or use text-to-speech software',
        'Use music and sound effects to enhance memory',
        'Join study groups with verbal interaction'
      ];
    }
    if (styleName.includes('DIGITAL')) {
      return [
        'Follow structured, step-by-step learning processes',
        'Use logical frameworks and systematic approaches',
        'Break down complex topics into smaller, manageable parts',
        'Create detailed outlines and organized notes',
        'Prefer sequential learning over random exploration'
      ];
    }
    if (styleName.includes('FLEXIBLE')) {
      return [
        'Adapt your learning style based on the situation',
        'Take on leadership roles in group learning',
        'Explore multiple approaches to solve problems',
        'Seek dynamic and changing learning environments',
        'Balance independent and collaborative learning'
      ];
    }
    return [
      'Combine multiple learning approaches for best results',
      'Focus on understanding concepts rather than memorization',
      'Seek feedback and adjust your learning strategies',
      'Create a comfortable learning environment',
      'Set clear goals and track your progress'
    ];
  };

  const StyleIcon = getStyleIcon(styleName);
  const styleColor = getStyleColor(styleName);
  const tips = getStyleTips(styleName);

  return (
    <div className="bg-white rounded-xl shadow-lg p-8">
      <div className="text-center mb-8">
        <div className="flex items-center justify-center space-x-2 mb-4">
          <BookOpen className="h-8 w-8 text-purple-600" />
          <h3 className="text-2xl font-bold text-gray-800">Learning Style</h3>
        </div>
        <p className="text-gray-600">
          Optimize your learning approach based on your root number {rootNumber}
        </p>
      </div>

      {/* Learning Style Header */}
      <div className={`mb-8 p-8 bg-gradient-to-r ${styleColor} rounded-xl text-white text-center`}>
        <div className="flex items-center justify-center space-x-4 mb-4">
          <div className="p-3 bg-white bg-opacity-20 rounded-full">
            <StyleIcon className="h-8 w-8" />
          </div>
          <div>
            <h4 className="text-2xl font-bold">{styleName}</h4>
            <p className="opacity-90">Root Number {rootNumber}</p>
          </div>
        </div>
        <p className="text-lg opacity-95">
          {approach}
        </p>
      </div>

      {/* Learning Tips */}
      <div className="mb-8">
        <h4 className="text-xl font-semibold text-gray-800 mb-6 flex items-center space-x-2">
          <Zap className="h-6 w-6 text-yellow-500" />
          <span>Personalized Learning Tips</span>
        </h4>
        
        <div className="grid gap-4">
          {tips.map((tip, index) => (
            <div
              key={index}
              className="flex items-start space-x-3 p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors duration-200"
            >
              <div className="w-6 h-6 bg-purple-600 text-white rounded-full flex items-center justify-center text-sm font-bold flex-shrink-0 mt-0.5">
                {index + 1}
              </div>
              <p className="text-gray-700 leading-relaxed">{tip}</p>
            </div>
          ))}
        </div>
      </div>

      {/* Additional Insights */}
      <div className="bg-gradient-to-r from-purple-50 to-blue-50 rounded-lg p-6 border border-purple-200">
        <h4 className="text-lg font-semibold text-gray-800 mb-3">Additional Insights</h4>
        <div className="space-y-3 text-sm">
          <p className="text-gray-700">
            <span className="font-medium">Best Environment:</span> Your {styleName.toLowerCase()} nature thrives 
            in environments that support {approach.toLowerCase()}.
          </p>
          <p className="text-gray-700">
            <span className="font-medium">Career Development:</span> Apply these learning preferences when 
            choosing training programs, professional development, or skill-building activities.
          </p>
          <p className="text-gray-700">
            <span className="font-medium">Team Collaboration:</span> Understanding your learning style helps 
            you communicate more effectively with colleagues and contribute to team projects.
          </p>
        </div>
      </div>
    </div>
  );
};

export default LearningStyle;