import React from 'react';
import { CalculationResult } from '../../services/api';
import { Save, CheckCircle, AlertCircle, User, Calendar, Briefcase } from 'lucide-react';
import PyramidGrid from './PyramidGrid';
import RootAnalysis from './RootAnalysis';
import SWOTAnalysis from './SWOTAnalysis';
import BusinessGuidance from './BusinessGuidance';
import LearningStyle from './LearningStyle';

interface CalculationResultsProps {
  result: CalculationResult;
  onSave?: () => void;
  isSaving?: boolean;
  saveError?: string;
  saveSuccess?: boolean;
  showSaveButton?: boolean;
}

const CalculationResults: React.FC<CalculationResultsProps> = ({
  result,
  onSave,
  isSaving = false,
  saveError,
  saveSuccess = false,
  showSaveButton = false,
}) => {
  const { calculation, interpretation, swot_combinations, business_guidance, learning_style } = result;

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div>
            <h2 className="text-3xl font-bold text-gray-800 mb-2">Your Numerology Analysis</h2>
            <div className="flex flex-wrap items-center gap-4 text-sm text-gray-600">
              <div className="flex items-center space-x-2">
                <User className="h-4 w-4" />
                <span>{calculation.name}</span>
              </div>
              <div className="flex items-center space-x-2">
                <Calendar className="h-4 w-4" />
                <span>{new Date(calculation.birthdate).toLocaleDateString()}</span>
              </div>
              {calculation.industry && (
                <div className="flex items-center space-x-2">
                  <Briefcase className="h-4 w-4" />
                  <span>{calculation.industry}</span>
                </div>
              )}
            </div>
          </div>

          {/* Save Button */}
          {showSaveButton && (
            <div className="flex flex-col items-end space-y-2">
              <button
                onClick={onSave}
                disabled={isSaving}
                className="flex items-center space-x-2 px-6 py-3 bg-green-600 text-white font-medium rounded-md hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-300"
              >
                {isSaving ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                    <span>Saving...</span>
                  </>
                ) : (
                  <>
                    <Save className="h-5 w-5" />
                    <span>Save Calculation</span>
                  </>
                )}
              </button>

              {/* Save Status */}
              {saveSuccess && (
                <div className="flex items-center space-x-1 text-green-600 text-sm">
                  <CheckCircle className="h-4 w-4" />
                  <span>Saved successfully!</span>
                </div>
              )}
              
              {saveError && (
                <div className="flex items-center space-x-1 text-red-600 text-sm">
                  <AlertCircle className="h-4 w-4" />
                  <span>{saveError}</span>
                </div>
              )}
            </div>
          )}
        </div>
      </div>

      {/* Pyramid Grid */}
      <PyramidGrid calculation={calculation} />

      {/* Root Number Analysis */}
      <RootAnalysis 
        rootNumber={calculation.root_number} 
        interpretation={interpretation} 
      />

      {/* SWOT Analysis */}
      {swot_combinations && swot_combinations.length > 0 && (
        <SWOTAnalysis combinations={swot_combinations} />
      )}

      {/* Business Guidance */}
      {business_guidance && (
        <BusinessGuidance 
          guidance={business_guidance} 
          industry={calculation.industry}
        />
      )}

      {/* Learning Style */}
      {learning_style && (
        <LearningStyle 
          style={learning_style} 
          rootNumber={calculation.root_number}
        />
      )}

      {/* Summary Card */}
      <div className="bg-gradient-to-r from-purple-600 to-blue-600 rounded-xl shadow-lg p-8 text-white">
        <h3 className="text-2xl font-bold mb-4">Your Numerology Summary</h3>
        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
          <div className="text-center">
            <div className="text-3xl font-bold">{calculation.root_number}</div>
            <div className="text-sm opacity-90">Root Number</div>
          </div>
          <div className="text-center">
            <div className="text-3xl font-bold">{interpretation?.element || 'N/A'}</div>
            <div className="text-sm opacity-90">Element</div>
          </div>
          <div className="text-center">
            <div className="text-3xl font-bold">{swot_combinations?.length || 0}</div>
            <div className="text-sm opacity-90">SWOT Patterns</div>
          </div>
          <div className="text-center">
            <div className="text-3xl font-bold">
              {business_guidance?.element_category?.split('/')[0] || 'N/A'}
            </div>
            <div className="text-sm opacity-90">Category</div>
          </div>
        </div>
        
        {interpretation && (
          <div className="mt-6 p-4 bg-white bg-opacity-20 rounded-lg">
            <h4 className="font-semibold mb-2">{interpretation.title}</h4>
            <p className="text-sm opacity-90 line-clamp-3">
              {interpretation.strengths}
            </p>
          </div>
        )}
      </div>
    </div>
  );
};

export default CalculationResults;