import React from 'react';
import { NumerologyCalculation } from '../../services/api';

interface PyramidGridProps {
  calculation: NumerologyCalculation;
}

const PyramidGrid: React.FC<PyramidGridProps> = ({ calculation }) => {
  const { L5, L6, L7, L8, L9, root_number } = calculation;

  // Create the pyramid grid structure (4 rows x 7 columns)
  const createGrid = () => {
    // Initialize with null values
    const grid = Array(4).fill(null).map(() => Array(7).fill(null));

    // L6 (row 0): 4 elements centered
    grid[0][1] = L6[0];
    grid[0][2] = L6[1];
    grid[0][4] = L6[2];
    grid[0][5] = L6[3];

    // L7 (row 1): 2 elements centered
    grid[1][2] = L7[0];
    grid[1][4] = L7[1];

    // L8 (row 2): 7 elements
    for (let i = 0; i < 7; i++) {
      grid[2][i] = L8[i];
    }

    // L9 (row 3): 3 elements centered
    grid[3][1] = L9[0];
    grid[3][3] = L9[1];
    grid[3][5] = L9[2];

    return grid;
  };

  const grid = createGrid();

  const getCellStyle = (value: number | null, row: number, col: number) => {
    if (value === null) {
      return 'invisible'; // Hidden cell
    }

    let baseStyle = 'w-12 h-12 flex items-center justify-center text-lg font-bold rounded-lg shadow-md transition-all duration-300 hover:transform hover:scale-110';

    // Highlight root number (L8[3] = grid[2][3])
    if (row === 2 && col === 3 && value === root_number) {
      baseStyle += ' bg-purple-600 text-white ring-4 ring-purple-200';
    } else if (row === 0) {
      // L6 cells
      baseStyle += ' bg-blue-100 text-blue-800 hover:bg-blue-200';
    } else if (row === 1) {
      // L7 cells
      baseStyle += ' bg-green-100 text-green-800 hover:bg-green-200';
    } else if (row === 2) {
      // L8 cells
      baseStyle += ' bg-purple-100 text-purple-800 hover:bg-purple-200';
    } else if (row === 3) {
      // L9 cells
      baseStyle += ' bg-yellow-100 text-yellow-800 hover:bg-yellow-200';
    }

    return baseStyle;
  };

  return (
    <div className="bg-white rounded-xl shadow-lg p-8">
      <h3 className="text-2xl font-bold text-gray-800 mb-6 text-center">
        Numerology Pyramid
      </h3>

      {/* Pyramid Grid */}
      <div className="flex justify-center mb-8">
        <div className="grid grid-cols-7 gap-2 max-w-md">
          {grid.map((row, rowIndex) =>
            row.map((cell, colIndex) => (
              <div
                key={`${rowIndex}-${colIndex}`}
                className={getCellStyle(cell, rowIndex, colIndex)}
                title={cell !== null ? `Row ${rowIndex + 1}, Col ${colIndex + 1}: ${cell}` : ''}
              >
                {cell}
              </div>
            ))
          )}
        </div>
      </div>

      {/* Legend */}
      <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
        <div className="flex items-center space-x-2">
          <div className="w-6 h-6 bg-blue-100 rounded-md border border-blue-200"></div>
          <span className="text-sm text-gray-700">L6 Layer</span>
        </div>
        <div className="flex items-center space-x-2">
          <div className="w-6 h-6 bg-green-100 rounded-md border border-green-200"></div>
          <span className="text-sm text-gray-700">L7 Layer</span>
        </div>
        <div className="flex items-center space-x-2">
          <div className="w-6 h-6 bg-purple-100 rounded-md border border-purple-200"></div>
          <span className="text-sm text-gray-700">L8 Layer</span>
        </div>
        <div className="flex items-center space-x-2">
          <div className="w-6 h-6 bg-yellow-100 rounded-md border border-yellow-200"></div>
          <span className="text-sm text-gray-700">L9 Layer</span>
        </div>
      </div>

      {/* Root Number Highlight */}
      <div className="text-center">
        <div className="inline-flex items-center space-x-2 px-4 py-2 bg-purple-600 text-white rounded-lg">
          <span className="text-sm">Root Number:</span>
          <span className="text-xl font-bold">{root_number}</span>
        </div>
      </div>

      {/* Detailed Breakdown */}
      <div className="mt-8 bg-gray-50 rounded-lg p-6">
        <h4 className="text-lg font-semibold text-gray-800 mb-4">Calculation Breakdown</h4>
        <div className="grid md:grid-cols-2 lg:grid-cols-5 gap-4 text-sm">
          <div>
            <div className="font-medium text-gray-700 mb-1">L5 (Raw)</div>
            <div className="text-gray-600">[{L5.join(', ')}]</div>
          </div>
          <div>
            <div className="font-medium text-gray-700 mb-1">L6 (Reduced)</div>
            <div className="text-gray-600">[{L6.join(', ')}]</div>
          </div>
          <div>
            <div className="font-medium text-gray-700 mb-1">L7 (Pairs)</div>
            <div className="text-gray-600">[{L7.join(', ')}]</div>
          </div>
          <div>
            <div className="font-medium text-gray-700 mb-1">L8 (Complex)</div>
            <div className="text-gray-600">[{L8.join(', ')}]</div>
          </div>
          <div>
            <div className="font-medium text-gray-700 mb-1">L9 (Final)</div>
            <div className="text-gray-600">[{L9.join(', ')}]</div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default PyramidGrid;