import React from 'react';
import { BusinessGuidance as BusinessGuidanceType } from '../../services/api';
import { Building, TrendingUp, Lightbulb, CheckCircle, AlertCircle } from 'lucide-react';

interface BusinessGuidanceProps {
  guidance: BusinessGuidanceType;
  industry?: string;
}

const BusinessGuidance: React.FC<BusinessGuidanceProps> = ({ guidance, industry }) => {
  const { element_category, suitable_businesses, industry_specific } = guidance;

  const isIndustryAligned = industry && industry_specific && 
    industry_specific.includes('aligns well');

  return (
    <div className="bg-white rounded-xl shadow-lg p-8">
      <div className="text-center mb-8">
        <div className="flex items-center justify-center space-x-2 mb-4">
          <Building className="h-8 w-8 text-purple-600" />
          <h3 className="text-2xl font-bold text-gray-800">Business Guidance</h3>
        </div>
        <p className="text-gray-600">
          Personalized business recommendations based on your numerology element
        </p>
      </div>

      {/* Element Category */}
      <div className="mb-8 p-6 bg-gradient-to-r from-purple-50 to-blue-50 rounded-lg border border-purple-200">
        <div className="flex items-center space-x-3 mb-3">
          <div className="p-2 bg-purple-600 text-white rounded-lg">
            <TrendingUp className="h-5 w-5" />
          </div>
          <h4 className="text-xl font-semibold text-gray-800">Your Element Category</h4>
        </div>
        <div className="text-2xl font-bold text-purple-600 mb-2">{element_category}</div>
        <p className="text-gray-700">
          This element category shapes your natural business inclinations and optimal working environments.
        </p>
      </div>

      {/* Industry Alignment */}
      {industry && industry_specific && (
        <div className={`mb-8 p-6 rounded-lg border-2 ${
          isIndustryAligned 
            ? 'bg-green-50 border-green-200' 
            : 'bg-yellow-50 border-yellow-200'
        }`}>
          <div className="flex items-center space-x-3 mb-3">
            {isIndustryAligned ? (
              <CheckCircle className="h-6 w-6 text-green-600" />
            ) : (
              <AlertCircle className="h-6 w-6 text-yellow-600" />
            )}
            <h4 className="text-lg font-semibold text-gray-800">
              Your Industry: {industry}
            </h4>
          </div>
          <p className={`${
            isIndustryAligned ? 'text-green-700' : 'text-yellow-700'
          }`}>
            {industry_specific}
          </p>
        </div>
      )}

      {/* Suitable Businesses */}
      <div className="mb-8">
        <div className="flex items-center space-x-3 mb-4">
          <div className="p-2 bg-blue-600 text-white rounded-lg">
            <Lightbulb className="h-5 w-5" />
          </div>
          <h4 className="text-xl font-semibold text-gray-800">Recommended Business Areas</h4>
        </div>
        
        <div className="bg-gray-50 rounded-lg p-6">
          <p className="text-gray-700 leading-relaxed mb-4">
            Based on your {element_category} nature, you are naturally suited for:
          </p>
          
          {/* Parse and display business areas as tags */}
          <div className="flex flex-wrap gap-2">
            {suitable_businesses.split(',').map((business, index) => (
              <span
                key={index}
                className="inline-block px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm font-medium hover:bg-blue-200 transition-colors duration-200"
              >
                {business.trim()}
              </span>
            ))}
          </div>
        </div>
      </div>

      {/* Tips Section */}
      <div className="bg-gradient-to-r from-gray-50 to-gray-100 rounded-lg p-6">
        <h4 className="text-lg font-semibold text-gray-800 mb-4">Business Success Tips</h4>
        <div className="space-y-3 text-sm">
          <div className="flex items-start space-x-2">
            <div className="w-2 h-2 bg-purple-600 rounded-full mt-2 flex-shrink-0"></div>
            <p className="text-gray-700">
              Focus on industries and roles that align with your {element_category} characteristics for natural success.
            </p>
          </div>
          <div className="flex items-start space-x-2">
            <div className="w-2 h-2 bg-purple-600 rounded-full mt-2 flex-shrink-0"></div>
            <p className="text-gray-700">
              Consider partnerships or collaborations that complement your elemental strengths.
            </p>
          </div>
          <div className="flex items-start space-x-2">
            <div className="w-2 h-2 bg-purple-600 rounded-full mt-2 flex-shrink-0"></div>
            <p className="text-gray-700">
              Use your numerology insights to make informed decisions about career pivots or business expansions.
            </p>
          </div>
          {!isIndustryAligned && industry && (
            <div className="flex items-start space-x-2">
              <div className="w-2 h-2 bg-yellow-600 rounded-full mt-2 flex-shrink-0"></div>
              <p className="text-gray-700">
                While your current industry may not be the perfect elemental match, you can still succeed by 
                leveraging your element's strengths within your field.
              </p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default BusinessGuidance;